<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use App\Models\User;

class LoginController extends Controller
{
    public function showLoginForm($slug = null)
    {
        // Verify Slug (Security Feature)
        // Assume first user is admin/owner.
        $admin = User::first();
        if ($admin && $slug !== $admin->login_slug) {
            abort(404);
        }

        // Redirect if already logged in
        if (Auth::check()) {
            return redirect()->route('analytics.index');
        }

        return view('auth.login');
    }

    public function login(Request $request)
    {
        $credentials = $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        $remember = $request->has('remember');

        if (Auth::attempt($credentials, $remember)) {
            $request->session()->regenerate();
            $user = Auth::user();

            // Security Question Check
            if ($user->is_security_question_enabled) {
                // Logout temporary, store user id
                Auth::logout();
                session(['auth.security_user_id' => $user->id]);
                session(['auth.remember' => $remember]);
                
                return redirect()->route('login.challenge');
            }

            // Normal Login Notify
            $this->sendLoginNotification($request, $user);

            return redirect()->intended(route('analytics.index'))
                ->with('success', 'Login berhasil! Selamat datang kembali.');
        }

        return back()->withErrors([
            'email' => 'Email atau password salah.',
        ])->onlyInput('email');
    }

    public function showChallengeForm()
    {
        if (!session()->has('auth.security_user_id')) {
            return redirect()->route('home');
        }
        
        $userId = session('auth.security_user_id');
        $user = User::find($userId);
        
        if (!$user) return redirect()->route('home');

        return view('auth.challenge', ['question' => $user->security_question]);
    }

    public function verifyChallenge(Request $request)
    {
        if (!session()->has('auth.security_user_id')) {
            return redirect()->route('home');
        }

        $userId = session('auth.security_user_id');
        $user = User::findOrFail($userId);

        $request->validate([
            'security_answer' => 'required|string',
        ]);

        // Normalize input: lowercase and trimmed
        $input = strtolower(trim($request->security_answer));

        if (Hash::check($input, $user->security_answer)) {
            // Success
            Auth::login($user, session('auth.remember', false));
            session()->forget(['auth.security_user_id', 'auth.remember']);
            $request->session()->regenerate();
            
            $this->sendLoginNotification($request, $user);

            return redirect()->intended(route('analytics.index'))
                 ->with('success', 'Verifikasi berhasil. Selamat datang!');
        }

        return back()->withErrors(['security_answer' => 'Jawaban tidak cocok.']);
    }

    protected function sendLoginNotification(Request $request, $user)
    {
        $userAgent = $request->userAgent();
        
        // Simple detection
        $device = preg_match('/mobile/i', $userAgent) ? 'Mobile' : 'Desktop';
        $browser = 'Unknown';
        if (preg_match('/Chrome/i', $userAgent)) $browser = 'Chrome';
        elseif (preg_match('/Firefox/i', $userAgent)) $browser = 'Firefox';
        elseif (preg_match('/Safari/i', $userAgent)) $browser = 'Safari';
        elseif (preg_match('/Edge/i', $userAgent)) $browser = 'Edge';
        
        $agentInfo = "$device - $browser";
        $user->notify(new \App\Notifications\LoginNotification($agentInfo, $request->ip(), now()));
    }

    public function logout(Request $request)
    {
        Auth::logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('home')
            ->with('success', 'Logout berhasil. Sampai jumpa lagi!');
    }
}
